/*
 * Copyright 2021 Tomas Zeman <tomas@functionals.cz>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package requestor4s

import akka.actor.typed.ActorSystem
import akka.http.scaladsl.Http
import akka.http.scaladsl.model.Uri.Query
import akka.http.scaladsl.model._
import akka.http.scaladsl.model.headers.Authorization
import requestor4s.Pickler._
import requestor4s.Session.Params

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

case class RequestorApi(auth: Authorization,
  baseUrl: String,
  timeout: FiniteDuration)(
  implicit system: ActorSystem[_], ec: ExecutionContext)
  extends Session[Future] with CustomerApi[Future] with AccountApi[Future]
  with TagApi[Future] {

  override def get[T](path: String, params: Params)(
    implicit rdr: Pickler.Reader[T]): Future[T] =
    for {
      f <- Http().singleRequest(HttpRequest(
        uri = Uri(s"$baseUrl/$path").withQuery(Query(params)),
        headers = Seq(auth)))
      b <- f.entity.toStrict(timeout)
    } yield read[T](b.data.utf8String)

  override def post[In, Out](path: String, params: Params, data: In)(
    implicit wr: Pickler.Writer[In], rdr: Pickler.Reader[Out]): Future[Out] =
    for {
      f <- Http().singleRequest(HttpRequest(
        method = HttpMethods.POST,
        uri = Uri(s"$baseUrl/$path").withQuery(Query(params)),
        headers = Seq(auth),
        entity = HttpEntity(ContentTypes.`application/json`, write(data))))
      b <- f.entity.toStrict(timeout)
    } yield read[Out](b.data.utf8String)

  override def postUnit[In](path: String, params: Params, data: In)(
    implicit wr: Pickler.Writer[In]): Future[Unit] =
    for {
      f <- Http().singleRequest(HttpRequest(
        method = HttpMethods.POST,
        uri = Uri(s"$baseUrl/$path").withQuery(Query(params)),
        headers = Seq(auth),
        entity = HttpEntity(ContentTypes.`application/json`, write(data))))
        _ <- f.entity.toStrict(timeout)
    } yield ()

}

