/*
 * Copyright 2021 Tomas Zeman <tomas@functionals.cz>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package requestor4s

trait AccountApi[F[_]] extends Session[F] {

  private val pfx = "Account"

  protected trait PathBuilder {
    lazy val path =
      s"$pfx/${getClass.getName.stripSuffix("$").split("\\$").reverse.head}"
  }

  object GetUsers extends PathBuilder {
    def apply(pageIndex: Int = 0, pageSize: Int = 200): F[GetUsersResult] =
      get[GetUsersResult](path, Map(
        "pageIndex" -> pageIndex.toString,
        "pageSize" -> pageSize.toString
      ))

    def apply(filter: UsersFilterPost): F[GetUsersResult] =
      post[UsersFilterPost, GetUsersResult](path, data = filter)
  }

  object GetRequestorUser extends PathBuilder {
    def me: F[GetRequestorUserResult] = get[GetRequestorUserResult](path)

    def apply(userNameOrUserProviderKey: String): F[GetRequestorUserResult] =
      get[GetRequestorUserResult](s"$path/$userNameOrUserProviderKey")
  }

  object CreateRequestorUser extends PathBuilder {
    def apply(user: CreateRequestorUserPost): F[CreateRequestorUserResult] =
      post[CreateRequestorUserPost, CreateRequestorUserResult](path,
        data = user)
  }

  object UpdateRequestorUser extends PathBuilder {
    def apply(user: UpdateRequestorUserPost): F[UpdateRequestorUserResult] =
      post[UpdateRequestorUserPost, UpdateRequestorUserResult](path,
        data = user)
  }

  object DeleteRequestorUser extends PathBuilder {
    def apply(userProviderKey: String, force: Boolean): F[Unit] =
      postUnit[ujson.Obj](path, data = ujson.Obj(
        "UserProviderKey" -> userProviderKey,
        "Force" -> force
      ))
  }

  object Account {
    protected trait TagOp extends PathBuilder {
      def apply(tagId: Int, userProviderKey: String): F[Unit] =
        postUnit[ujson.Obj](path, data = ujson.Obj(
          "TagId" -> tagId, "UserProviderKey" -> userProviderKey
        ))
    }
    object AddTag extends TagOp
    object RemoveTag extends TagOp
  }
}
