/*
 * Copyright 2021 Tomas Zeman <tomas@functionals.cz>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package requestor4s

trait CustomerApi[F[_]] extends Session[F] {

  private val pfx = "Customer"

  protected trait PathBuilder {
    lazy val path =
      s"$pfx/${getClass.getName.stripSuffix("$").split("\\$").reverse.head}"
  }

  object GetCustomer extends PathBuilder {
    def apply(id: Int): F[Customer] = get[Customer](s"$path/$id")
  }

  object GetCustomers extends PathBuilder {
    def apply(pageIndex: Int = 0, pageSize: Int = 200): F[GetCustomersResult] =
      get[GetCustomersResult](path, Map(
        "pageIndex" -> pageIndex.toString,
        "pageSize" -> pageSize.toString
      ))
  }

  object InsertCustomer extends PathBuilder {
    def apply(c: Customer): F[IdResult] =
      post[Customer, IdResult](path, data = c)
  }

  object UpdateCustomer extends PathBuilder {
    def apply(c: Customer): F[Unit] =
      postUnit[Customer](s"$path/${c.Id}", data = c)
  }

  object DeleteCustomer extends PathBuilder {

    def apply(id: Int): F[Unit] = {
      postUnit[ujson.Obj](s"$path/$id", data = ujson.Obj())
    }

    def apply(c: Customer): F[Unit] = apply(c.Id)
  }

  object Customer {
    protected trait TagOp extends PathBuilder {
      def apply(tagId: Int, customerId: Int): F[Unit] =
        postUnit[ujson.Obj](path, data = ujson.Obj(
          "TagId" -> tagId, "CustomerId" -> customerId
        ))
    }
    object AddTag extends TagOp
    object RemoveTag extends TagOp
  }
}
